/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

/* ATMOSPHERIC CONSTANTS */

const float kilo = 1e3;

const float planetRadius          = 6371e3;
const float atmosphereLowerRadius = planetRadius;
const float atmosphereUpperRadius = planetRadius + 110e3;

const vec2 scaleHeights = vec2(8.40, 1.25) * kilo;

const float mieScatteringAlbedo = 0.9;
const float mieAnisotropyFactor = 2800.0; // e asymmetry term

const float turbidity             = 1.1;
const float airNumberDensity      = 2.5035422e25;
const float atmosphereTemperature = 293.15; // Kelvins
const float ozonePeakDensity      = 5e-6;
const float ozonePeakAltitude     = 35e3;
const float ozoneNumberDensity    = airNumberDensity * exp(-ozonePeakAltitude / 8e3) * (134.628 / 48.0) * ozonePeakDensity;
const float ozoneUnitConversion   = 1e-4; // Converts from cm² to m²

const vec3 rayleighScatteringCoefficients = vec3(6.42905682e-6, 1.08663713e-5, 2.4844733e-5);
const vec3 mieScatteringCoefficients      = vec3(21e-6);
const vec3 ozoneExtinctionCoefficients    = vec3(4.51103766177301e-21, 3.2854797958699e-21, 1.96774621921165e-22) * ozoneNumberDensity * ozoneUnitConversion;

#if TONEMAP == ACES
    const mat2x3 atmosphereScatteringCoefficients  = mat2x3(rayleighScatteringCoefficients * SRGB_2_AP1_ALBEDO,  mieScatteringCoefficients * SRGB_2_AP1_ALBEDO);
    const mat3x3 atmosphereAttenuationCoefficients = mat3x3(rayleighScatteringCoefficients * SRGB_2_AP1_ALBEDO, (mieScatteringCoefficients * SRGB_2_AP1_ALBEDO) / mieScatteringAlbedo, ozoneExtinctionCoefficients * SRGB_2_AP1_ALBEDO);
#else
    const mat2x3 atmosphereScatteringCoefficients  = mat2x3(rayleighScatteringCoefficients, mieScatteringCoefficients);
    const mat3x3 atmosphereAttenuationCoefficients = mat3x3(rayleighScatteringCoefficients, mieScatteringCoefficients / mieScatteringAlbedo, ozoneExtinctionCoefficients);
#endif

vec3 atmosphereRayPosition = vec3(0.0, planetRadius, 0.0) + cameraPosition;

/* CLOUDS CONSTANTS */

const float cloudsExtinctionCoefficient = 0.08;
const float cloudsScatteringCoefficient = 0.99;
const float cloudsTransmitThreshold     = 0.05;

const float cloudsForwardsLobe = 0.80;
const float cloudsBackardsLobe = 0.25;
const float cloudsForwardsPeak = 0.85;
const float cloudsBackScatter  = 0.20;
const float cloudsPeakWeight   = 0.10;

const int   cloudsMultiScatterSteps = 8;
const float cloudsExtinctionFalloff = 0.60;
const float cloudsScatteringFalloff = 0.50;
const float cloudsAnisotropyFalloff = 0.80;

/* CELESTIAL CONSTANTS */

const float sunTemperature = 5778.0; // Kelvins
const float sunRadius      = 6.9634e8;
const float sunDistance    = 1.496e11;

const float moonRadius    = 1.7374e6;
const float moonDistance  = 3.8440e8;
const float moonAlbedo    = 0.14; // The moon reflects approximately 7-14% of the sun's emitted light 
const float moonRoughness = 0.40;

const float sunAngularRadius  = CELESTIAL_SIZE_MULTIPLIER * sunRadius  / sunDistance;
      float moonAngularRadius = CELESTIAL_SIZE_MULTIPLIER * moonRadius / moonDistance;

const vec3 sunIrradiance = vec3(1.0, 0.949, 0.937) * 126e3; // Brightness of light reaching the earth (~126'000 lux)
      vec3 sunRadiance   = sunIrradiance / coneAngleToSolidAngle(sunAngularRadius);

vec3 moonRadiance   = moonAlbedo   * sunIrradiance;
vec3 moonIrradiance = moonRadiance * coneAngleToSolidAngle(moonAngularRadius); // The rough amount of light the moon emits that reaches the earth

float shadowLightAngularRadius = sunAngle < 0.5 ? sunAngularRadius : moonAngularRadius;
