/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

/*
    [Credits]:
        Jessie     - help with atmospheric scattering and providing ozone cross section approximation (https://github.com/Jessie-LC)
        sixthsurge - providing uniform sphere sampling method (https://github.com/sixthsurge)
        
    [References]:
        Nishita, T. (1993). Display of the earth taking into account atmospheric scattering. http://nishitalab.org/user/nis/cdrom/sig93_nis.pdf
        Elek, O. (2009). Rendering Parametrizable Planetary Atmospheres with Multiple Scattering in Real-Time. https://old.cescg.org/CESCG-2009/papers/PragueCUNI-Elek-Oskar09.pdf
        Novak, J. (2014). Residual Ratio Tracking for Estimating Attenuation in Participating Media. https://cs.dartmouth.edu/~wjarosz/publications/novak14residual.pdf
*/

#include "/include/atmospherics/constants.glsl"

#include "/include/utility/phase.glsl"

vec3 getAtmosphereDensities(float centerDist) {
	float altitudeKm = (centerDist - planetRadius) * 1e-3;
	vec2 rayleighMie = exp(altitudeKm / -(scaleHeights * 1e-3));

    // Ozone approximation from Jessie#7257
    float o1 = 25.0 *     exp(( 0.0 - altitudeKm) * rcp(  8.0));
    float o2 = 30.0 * pow(exp((18.0 - altitudeKm) * rcp( 80.0)), altitudeKm - 18.0);
    float o3 = 75.0 * pow(exp((25.3 - altitudeKm) * rcp( 35.0)), altitudeKm - 25.3);
    float o4 = 50.0 * pow(exp((30.0 - altitudeKm) * rcp(150.0)), altitudeKm - 30.0);
    float ozone = (o1 + o2 + o3 + o4) * rcp(134.628);

	return vec3(rayleighMie, ozone);
}

vec3 evaluateAtmosphereTransmittance(vec3 rayOrigin, vec3 lightDir) {
    float stepSize   = intersectSphere(rayOrigin, lightDir, atmosphereUpperRadius).y * rcp(ATMOSPHERE_TRANSMITTANCE_STEPS);
    vec3 increment   = lightDir * stepSize;
    vec3 rayPosition = rayOrigin + increment * 0.5;

    vec3 accumAirmass = vec3(0.0);
    for(int i = 0; i < ATMOSPHERE_TRANSMITTANCE_STEPS; i++, rayPosition += increment) {
        accumAirmass += getAtmosphereDensities(length(rayPosition)) * stepSize;
    }
    return exp(-atmosphereAttenuationCoefficients * accumAirmass);
}

#if defined STAGE_FRAGMENT

    vec3 evaluateAtmosphericScattering(vec3 rayDirection, vec3 skyIlluminance) {
        vec2 dists = intersectSphericalShell(atmosphereRayPosition, rayDirection, atmosphereLowerRadius - 2e3, atmosphereUpperRadius);
        if(dists.y < 0.0) return vec3(0.0);

        float stepSize   = (dists.y - dists.x) * rcp(ATMOSPHERE_SCATTERING_STEPS);
        vec3 increment   = rayDirection * stepSize;
        vec3 rayPosition = atmosphereRayPosition + increment * 0.5;

        vec2 VdotL = vec2(dot(rayDirection, sunVector), dot(rayDirection, moonVector));
        vec4 phase = vec4(
            vec2(rayleighPhase(VdotL.x), kleinNishinaPhase(VdotL.x, mieAnisotropyFactor)), 
            vec2(rayleighPhase(VdotL.y), kleinNishinaPhase(VdotL.y, mieAnisotropyFactor))
        );

        mat2x3 scattering = mat2x3(vec3(0.0), vec3(0.0)); vec3 multipleScattering = vec3(0.0); vec3 transmittance = vec3(1.0);
    
        for(int i = 0; i < ATMOSPHERE_SCATTERING_STEPS; i++, rayPosition += increment) {
            vec3 airmass          = getAtmosphereDensities(length(rayPosition)) * stepSize;
            vec3 stepOpticalDepth = atmosphereAttenuationCoefficients * airmass;

            vec3 stepTransmittance  = exp(-stepOpticalDepth);
            vec3 visibleScattering  = transmittance * saturate((stepTransmittance - 1.0) / -stepOpticalDepth);

            vec3 sunStepScattering  = atmosphereScatteringCoefficients * (airmass.xy * phase.xy) * visibleScattering;
            vec3 moonStepScattering = atmosphereScatteringCoefficients * (airmass.xy * phase.zw) * visibleScattering;

            scattering[0] += sunStepScattering  * evaluateAtmosphereTransmittance(rayPosition, sunVector );
            scattering[1] += moonStepScattering * evaluateAtmosphereTransmittance(rayPosition, moonVector);

            vec3 stepScattering    = atmosphereScatteringCoefficients * airmass.xy;
            vec3 stepScatterAlbedo = stepScattering / stepOpticalDepth;

            vec3 multScatteringFactor = stepScatterAlbedo * 0.84;
            vec3 multScatteringEnergy = multScatteringFactor / (1.0 - multScatteringFactor);
                 multipleScattering  += multScatteringEnergy * visibleScattering * stepScattering;

            transmittance *= stepTransmittance;
        }
        multipleScattering *= skyIlluminance * isotropicPhase;
        scattering[0]      *= sunIrradiance;
        scattering[1]      *= moonIrradiance;
    
        return scattering[0] + scattering[1] + multipleScattering;
    }

#endif

vec3 evaluateDirectIlluminance() {
    vec3 directIlluminance = vec3(0.0);

    #if defined WORLD_OVERWORLD
        directIlluminance += evaluateAtmosphereTransmittance(atmosphereRayPosition, sunVector ) * sunIrradiance;
        directIlluminance += evaluateAtmosphereTransmittance(atmosphereRayPosition, moonVector) * moonIrradiance;
    #endif
    return max0(directIlluminance);
}

vec3 evaluateUniformSkyIrradiance() {
    vec3 skyIlluminance = vec3(0.0);

    #if defined WORLD_OVERWORLD || defined WORLD_END
        const ivec2 samples = ivec2(8, 8);

        for(int x = 0; x < samples.x; x++) {
            for(int y = 0; y < samples.y; y++) {
                vec3 direction  = generateUnitVector((vec2(x, y) + 0.5) / samples);
                skyIlluminance += texture(ATMOSPHERE_BUFFER, projectSphere(direction)).rgb;
            }
        }
        skyIlluminance *= PI / (samples.x * samples.y);
    #endif
    return max0(skyIlluminance);
}
