/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

/*
    [Credits]
        Jessie - providing hemispherical albedo function (https://github.com/Jessie-LC)

    [References]:
        Karis, B. (2013). Specular BRDF Reference. http://graphicrants.blogspot.com/2013/08/specular-brdf-reference.html
        de Carpentier, G., & Ishiyama, K. (2017). DECIMA ENGINE: ADVANCES IN LIGHTING AND AA. https://www.guerrilla-games.com/media/News/Files/DecimaSiggraph2017.pdf
        Heitz, E. (2017). A Simpler and Exact Sampling Routine for the GGX Distribution of Visible Normals. https://hal.science/hal-01509746/document
        Dupuy, J., & Benyoub, A. (2024). Sampling Visible GGX Normals with Spherical Caps. https://arxiv.org/pdf/2306.05044.pdf
*/

//////////////////////////////////////////////////////////
/*------------------ GGX DISTRIBUTION ------------------*/
//////////////////////////////////////////////////////////

float distribution_GGX(float cosTheta, float alphaSq) {
    float denom = cosTheta * cosTheta * (alphaSq - 1.0) + 1.0;
    return alphaSq / (PI * denom * denom);
}

float lambda_Smith(float cosTheta, float alphaSq) {
    float cosThetaSq = cosTheta * cosTheta;
    return (-1.0 + sqrt(1.0 + alphaSq * (1.0 - cosThetaSq) / cosThetaSq)) * 0.5;
}

//////////////////////////////////////////////////////////
/*-------------------- MICROSURFACE --------------------*/
//////////////////////////////////////////////////////////

float smithG_GGX(float cosTheta, float alphaSq) {
    float cosThetaSq = cosTheta * cosTheta;
    return 1.0 / maxEps(cosTheta + sqrt(alphaSq + cosThetaSq - alphaSq * cosThetaSq));
}

float G1_Smith_GGX(float cosTheta, float alphaSq) {
    return 1.0 / (1.0 + lambda_Smith(cosTheta, alphaSq));
}

float G2_Smith_Height_Correlated(float NdotV, float NdotL, float alphaSq) {
    float lambdaV = lambda_Smith(NdotV, alphaSq);
    float lambdaL = lambda_Smith(NdotL, alphaSq);
    return 1.0 / (1.0 + lambdaV + lambdaL);
}

float G2_Smith_Separable(float NdotV, float NdotL, float alphaSq) {
    return smithG_GGX(NdotV, alphaSq) * smithG_GGX(NdotL, alphaSq);
}

vec3 sampleGGXVNDF(vec3 viewDirection, vec2 xi, vec2 alpha) {
    viewDirection = normalize(vec3(alpha * viewDirection.xy, viewDirection.z));

    float phi       = TAU * xi.x;
    float cosTheta  = (1.0 - xi.y) * (1.0 + viewDirection.z) - viewDirection.z;
    float sinTheta  = sqrt(saturate(1.0 - cosTheta * cosTheta));
    vec3  reflected = vec3(vec2(cos(phi), sin(phi)) * sinTheta, cosTheta);

    vec3 halfway = reflected + viewDirection;

    return normalize(vec3(alpha * halfway.xy, halfway.z));
}

vec3 evaluateMicrosurfaceOpaqueDiffuse(Material material, vec3 lightDirection) {
    float NdotL = maxEps(dot(material.normal, lightDirection));
    return material.albedo * NdotL * RCP_PI;
}

void sampleMicrosurfaceOpaqueDiffuse(inout vec3 throughput, inout vec3 wr, Material material) {
    vec3 eta = material.N / airIOR, etaK = material.K / airIOR;

    vec3 fresnel = fresnelDielectricConductor(dot(material.normal, -wr), eta, etaK);

    wr = generateCosineVector(material.normal, rand2F());

    throughput *= (1.0 - fresnel);
    throughput *= material.albedo * material.ao;
    throughput *= fresnelDielectricDielectric_T(dot(material.normal, wr), airIOR, material.N);
}

vec3 evaluateMicrosurfaceOpaqueSpecular(Material material, vec3 rayDirection, vec3 lightDirection) {
    float alphaSq = maxEps(material.alpha.r * material.alpha.r);

    vec3  halfway = normalize(rayDirection + lightDirection);
    float NdotL   = saturate(dot(material.normal, lightDirection));
    float NdotV   = saturate(dot(material.normal, rayDirection  ));
    float VdotH   = dot(rayDirection   , halfway);
    float NdotH   = dot(material.normal, halfway);
    
    float D  = distribution_GGX(NdotH, alphaSq);
    vec3  F  = fresnelDielectricConductor(VdotH, material.N, material.K);
    float G2 = G2_Smith_Height_Correlated(NdotV, NdotL, alphaSq);

    return NdotL * D * F * G2 / maxEps(4.0 * NdotV * NdotL);
}

void sampleMicrosurfaceOpaqueSpecular(inout vec3 throughput, inout vec3 wr, Material material) {
    vec3 eta = material.N / airIOR, etaK = material.K / airIOR;

    vec3 wi = -wr;
    vec3 wm = sampleGGXVNDF(wi, rand2F(), material.alpha);

    wr = reflect(wr, wm);

    float alphaSq = maxEps(material.alpha.r * material.alpha.r);

    vec3  halfway = normalize(wi + wr);
    float NdotL   = saturate(dot(wm, wr));
    float NdotV   = saturate(dot(wm, wi));
    float VdotH   = dot(wi, halfway);
    float NdotH   = dot(wm, halfway);

    float D  = distribution_GGX(NdotH, alphaSq);
    vec3  F  = fresnelDielectricConductor(VdotH, eta, etaK);
    float G1 = G1_Smith_GGX(NdotV, alphaSq);
    float G2 = G2_Smith_Height_Correlated(NdotV, NdotL, alphaSq);

    vec3 brdf = D * F * G2 / (4.0 * NdotV * NdotL);
    float pdf = D * G1     / (4.0 * NdotV);

    throughput *= brdf / pdf;
}
