/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

void pathtraceBelmuBRDFDiffuse(vec3 origin, vec3 direction, vec3 directIlluminance, inout vec3 irradiance, inout vec3 samplePosition, inout vec3 sampleNormal) {
    vec3 rayPosition  = origin;
    vec3 rayDirection = direction;
    
    VoxelIntersection hit;
    hit.position = origin;
    
    Material material = getRasterMaterial(textureCoords);

    rayPosition += material.geometricNormal * 1e-3;

    if (material.F0.r * maxFloat8 > 229.5) return;

    vec3 throughput = vec3(1.0);
    vec3 estimate   = vec3(0.0);

    for(int j = 0; j <= MAX_PT_BOUNCES; j++) {
        
        /* Russian Roulette */
        #if RUSSIAN_ROULETTE == 1
            float roulette = saturate(maxOf(throughput));
            if (roulette < randF()) {
                throughput = vec3(0.0);
                break;
            }
            throughput /= roulette;
        #endif

        bool firstBounce = j == 0;

        if (!firstBounce) {
            /* Voxel Tracing */
            hit = raytraceVoxel2(rayPosition, rayDirection, MAX_PT_BOUNCE_STEPS, true, true);

            /* Path Data */
            material = getVoxelMaterial(hit.voxel.packedData, hit.textureCoords, hit.position, hit.normal, 0.0);

            if (j == 1) {
                samplePosition = hit.position;
                sampleNormal   = material.normal;
            }

            /* Sky Contribution */
            if (!hit.intersect) {
                #if defined WORLD_OVERWORLD
                    vec3 atmosphere = texture(ATMOSPHERE_BUFFER, projectSphere(rayDirection)).rgb;
                    estimate += throughput * atmosphere;
                #endif
                break;
            }

            ivec3 cachePosition = ivec3(floor(hit.position - hit.normal * 1e-4));
            vec3  cacheEntry    = readCurrentCacheEntry(cachePosition, getFaceIndex(hit.normal));

            if (cacheEntry != vec3(0.0)) {
                estimate += throughput * cacheEntry;
            }

            rayPosition = hit.position + hit.normal * 1e-3;
        }

        /* Shadow Ray */
        if (!firstBounce) {
            #if defined WORLD_OVERWORLD
                vec3 shadowRayDirection = generateConeVector(shadowLightVector, rand2F(), shadowLightAngularRadius);
                
                vec3 brdf = evaluateMicrosurfaceOpaqueDiffuse(material, shadowRayDirection);

                float visibility = sampleVisibility(shadowtex0, rayPosition - cameraPosition);

                estimate += throughput * brdf * directIlluminance * visibility;
            #endif
        }

        /* Phase Sampling */
        vec3 phase = vec3(1.0);
        sampleMicrosurfaceOpaqueDiffuse(phase, rayDirection, material);

        if (dot(material.normal, rayDirection) <= 0.0) break;
        
        if (!firstBounce) {
            throughput *= phase;

            /* Material Emission */
            estimate += throughput * EMISSIVE_INTENSITY * EMISSIVE_INTENSITY_MULTIPLIER * material.emission;
        }
    }
    
    irradiance += estimate * rcp(MAX_PT_SAMPLES);
}

void pathtraceBelmuBRDFSpecular(vec3 origin, vec3 direction, vec3 directIlluminance, inout vec3 irradiance, inout vec3 samplePosition, inout vec3 sampleNormal) {
    Material material = getRasterMaterial(textureCoords);

    vec3 rayPosition  = origin + material.normal * 1e-2;
    vec3 rayDirection = direction;
    
    VoxelIntersection hit;
    hit.position = origin;

    vec3 throughput = vec3(1.0);
    vec3 estimate   = vec3(0.0);

    for(int j = 0; j <= MAX_PT_BOUNCES; j++) {
        
        /* Russian Roulette */
        #if RUSSIAN_ROULETTE == 1
            float roulette = saturate(maxOf(throughput));
            if (roulette < randF()) {
                throughput = vec3(0.0);
                break;
            }
            throughput /= roulette;
        #endif

        bool firstBounce = j == 0;

        if (!firstBounce) {
            /* Voxel Tracing */
            hit = raytraceVoxel2(rayPosition, rayDirection, MAX_PT_BOUNCE_STEPS, true, true);

            /* Path Data */
            material = getVoxelMaterial(hit.voxel.packedData, hit.textureCoords, hit.position, hit.normal, 0.0);
            
            if (j == 1) {
                samplePosition = hit.position;
                sampleNormal   = material.normal;
            }

            /* Sky Contribution */
            if (!hit.intersect) {
                #if defined WORLD_OVERWORLD
                    vec3 atmosphere = texture(ATMOSPHERE_BUFFER, projectSphere(rayDirection)).rgb;
                    estimate += throughput * atmosphere;
                #endif
                break;
            }

            ivec3 cachePosition = ivec3(floor(hit.position - hit.normal * 1e-4));
            vec3  cacheEntry    = readCurrentCacheEntry(cachePosition, getFaceIndex(hit.normal));

            if (cacheEntry != vec3(0.0)) {
                bool isMetal = material.F0.r * maxFloat8 > 229.5;

                if (!isMetal)
                    cacheEntry *= material.albedo;

                estimate += throughput * cacheEntry;
            }

            rayPosition = hit.position + hit.normal * 1e-3;
        }

        /* Shadow Ray */
        if (!firstBounce) {
            #if defined WORLD_OVERWORLD
                vec3 shadowRayDirection = generateConeVector(shadowLightVector, rand2F(), shadowLightAngularRadius);
                
                vec3 brdf = evaluateMicrosurfaceOpaqueSpecular(material, -rayDirection, shadowRayDirection);

                float visibility = sampleVisibility(shadowtex0, rayPosition - cameraPosition);

                estimate += throughput * brdf * directIlluminance * visibility;
            #endif
        }
        
        /* Material Emission */
        if (!firstBounce) estimate += throughput * material.albedo * EMISSIVE_INTENSITY * EMISSIVE_INTENSITY_MULTIPLIER * material.emission;

        /* Phase Sampling */
        vec3 phase = vec3(1.0);
        mat3 tbn   = getTBNMatrix(material.normal);

        rayDirection = rayDirection * tbn;
        sampleMicrosurfaceOpaqueSpecular(phase, rayDirection, material);
        rayDirection = tbn * rayDirection;

        if (dot(material.normal, rayDirection) <= 0.0) break;

        #if RESTIR_GI == 0
            throughput *= phase;
        #else
            if(!firstBounce) throughput *= phase;
        #endif
    }
    
    irradiance += estimate * rcp(MAX_PT_SAMPLES);
}
