/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

float computeDiffusePDF(vec3 samplePosition, Material material, vec3 visiblePosition, vec3 visibleNormal) {
	if(material.F0 * maxFloat8 > 229.5) return 0.0;

	vec3 lightDirection = normalize(samplePosition - visiblePosition);
	float NdotL = saturate(dot(visibleNormal, lightDirection));
	
	return NdotL * RCP_PI;
}

vec3 computeSpecularBRDF(Material material, vec3 rayDirection, vec3 lightDirection) {
	float alphaSq = maxEps(material.alpha.r * material.alpha.r);

	vec3  halfway = normalize(rayDirection + lightDirection);
	float NdotH   = dot(material.normal, halfway);
	float LdotH   = dot(lightDirection , halfway);
	float VdotH   = dot(rayDirection   , halfway);

    float D  = distribution_GGX(NdotH, alphaSq);
	vec3  F  = fresnelDielectricConductor(VdotH, material.N, material.K);
    float G2 = G2_Smith_Height_Correlated(VdotH, LdotH, alphaSq);

	return D * F * G2 / (4.0 * VdotH * LdotH);
}

float computeSpecularPDF(vec3 samplePosition, Material material, vec3 visiblePosition, vec3 visibleNormal) {
	vec3 rayDirection   = normalize(-(visiblePosition - cameraPosition));
	vec3 lightDirection = normalize(samplePosition - visiblePosition);

	float alphaSq = maxEps(material.alpha.r * material.alpha.r);

	vec3  halfway = normalize(rayDirection + lightDirection);
	float NdotH   = dot(material.normal, halfway);
	float VdotH   = dot(rayDirection   , halfway);

    float D  = distribution_GGX(NdotH, alphaSq);
	float G1 = G1_Smith_GGX(VdotH, alphaSq);

	return D * G1 / (4.0 * VdotH);
}

float computeIntegrand(vec3 sampleRadiance, vec3 samplePosition, Material material, vec3 visiblePosition, vec3 visibleNormal) {
	vec3 rayDirection   = normalize(-(visiblePosition - cameraPosition));
	vec3 lightDirection = normalize(samplePosition - visiblePosition);

	float diffuse  = saturate(dot(visibleNormal, lightDirection)) * RCP_PI;
	vec3  specular = computeSpecularBRDF(material, rayDirection, lightDirection);

	float isMetal = float(material.F0 * maxFloat8 > 229.5);

	return luminance(sampleRadiance * (diffuse * (1.0 - isMetal) + specular)); //diffuse * (1.0 - isMetal)
}

float balanceHeuristic(float n_i, float p_i, float n_k, float p_k) {
	float denominator = n_i * p_i + n_k * p_k;
	return denominator <= 0.0 ? 0.0 : max0(n_i * p_i) / denominator;
}

Reservoir pickReservoirMIS(Reservoir diffuse, Reservoir specular, Material material, vec3 visiblePosition, vec3 visibleNormal) {
	float source_d_d = computeDiffusePDF(diffuse.position , material, visiblePosition, visibleNormal);
	float source_d_s = computeDiffusePDF(specular.position, material, visiblePosition, visibleNormal);

	float source_s_d = computeSpecularPDF(diffuse.position , material, visiblePosition, visibleNormal);
	float source_s_s = computeSpecularPDF(specular.position, material, visiblePosition, visibleNormal);

	float diffuseWeight  = source_d_d / (source_d_d + source_s_d);
	float specularWeight = source_s_s / (source_s_s + source_d_s);

	float p_hat_diffuse = luminance(diffuse.radiance * computeDiffusePDF(diffuse.position, material, visiblePosition, visibleNormal));

	Reservoir reservoir;
	mergeReservoir(reservoir, diffuse , diffuse.weight  * diffuseWeight * p_hat_diffuse); 
	mergeReservoir(reservoir, specular, specular.weight * specularWeight);

	float p_hat = computeIntegrand(reservoir.radiance, reservoir.position, material, visiblePosition, visibleNormal);
	finalizeReservoirMIS(reservoir, p_hat, 1);

	return reservoir;
}

float computeJacobianDeterminant(vec3 samplePosition, vec3 sampleNormal, vec3 dstPosition, vec3 srcPosition) {
	vec3 offset_x1_x2 = samplePosition - srcPosition;
	vec3 offset_y1_x2 = samplePosition - dstPosition;

	float cosPhi_x1_x2 = abs(dot(sampleNormal, normalize(offset_x1_x2)));
	float cosPhi_y1_x2 = abs(dot(sampleNormal, normalize(offset_y1_x2)));

	float denominator = cosPhi_x1_x2 * lengthSqr(offset_y1_x2);
	float jacobian    = denominator == 0.0 ? 0.0 : (cosPhi_y1_x2 * lengthSqr(offset_x1_x2)) / denominator;

	return jacobian < 0.0 || isnan(jacobian) || isinf(jacobian) ? 0.0 : jacobian;
}
