/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

uniform sampler2D depthtex1;
uniform sampler2D depthtex2;
uniform sampler2D shadowcolor1;

#define ATLAS_ALBEDO   colortex7
#define ATLAS_NORMAL   depthtex2
#define ATLAS_SPECULAR shadowcolor1

struct Material {
    int id;

    #if IS_RGB
        vec3 albedo;
    #else
        float albedo;
    #endif

    float opacity;

    vec3 normal;
    vec3 tangentNormal;
    vec3 geometricNormal;

    #if IS_RGB
        vec3 N;
        vec3 K;
    #else
        float N;
        float K;
    #endif

    float F0;
    vec2  alpha;
    float ao;
    float emission;
    float subsurface;
    float porosity;
    float parallaxSelfShadowing;
};

#if RENDER_MODE == 0
    vec3 airIOR = vec3(1.00029);
#else
    float airIOR;
#endif

float f0ToIOR(float F0) {
	F0 = sqrt(F0) * 0.99999;
	return airIOR.r * ((1.0 + F0) / (1.0 - F0));
}

vec3 f0ToIOR(vec3 F0) {
	F0 = sqrt(F0) * 0.99999;
	return airIOR * ((1.0 + F0) / (1.0 - F0));
}

float iorToF0(float ior) {
	float a = (ior - airIOR.r) / (ior + airIOR.r);
	return a * a;
}

vec3 iorToF0(vec3 ior) {
	vec3 a = (ior - airIOR) / (ior + airIOR);
	return a * a;
}

#if RENDER_MODE == 1
    #include "/include/materials/dispersion.glsl"
    #include "/include/utility/spectral.glsl"
#endif

#include "/include/materials/hardcoded.glsl"
#include "/include/materials/gerstner.glsl"

#if RAIN_PUDDLES == 1
    #include "/include/materials/puddles.glsl"
#endif

mat3 calculateTBN(vec3 hitNormal) {
    vec3 tangent = mix(vec3(hitNormal.z, 0.0, -hitNormal.x), vec3(1.0, 0.0, 0.0), step(0.5, abs(hitNormal.y)));
    return mat3(tangent, -cross(hitNormal, tangent), hitNormal);
}

ivec2 calculateTextureCoords(uint packedTileData, vec3 hitPos, vec3 hitNormal) {
    float tileSize  =  exp2((packedTileData & 0xf) + 4);
    vec2  tileIndex =  vec2(uvec2(packedTileData) >> uvec2(4, 16) & 0xfff);

    vec3 absNormal   = abs(hitNormal);
    vec2 fractCoords = fract(absNormal.y > absNormal.x && absNormal.y > absNormal.z ? hitPos.xz : vec2(hitPos.x * sign(hitNormal.z) - hitPos.z * sign(hitNormal.x), -hitPos.y));
    return ivec2(tileIndex * tileSize + fractCoords * tileSize);
}

Material getVoxelMaterial(uint packedData, ivec2 textureCoords, vec3 hitPosition, vec3 hitNormal, float wavelength) {
    int  blockId = unpackId(packedData);
    vec3 tint    = unpackTint(packedData);

    #if IS_SPECTRAL
        switch(blockId) {
            case WATER:
                vec3 tangentNormal = getWaterNormals(hitPosition, WATER_OCTAVES);
                vec3 waveNormal    = normalize(calculateTBN(hitNormal) * tangentNormal);
                return getWaterMaterial(wavelength, waveNormal, tangentNormal, hitNormal);

            #if SPHERES_MODE == 1
                case SPHERE_0: return getMetalMaterial(wavelength,       hitNormal, vec2(METAL_SPHERE_ROUGHNESS));
                case SPHERE_1: return getGlassMaterial(wavelength, tint, hitNormal, vec2(GLASS_SPHERE_ROUGHNESS));
            #endif
            default: break;
        }
    #endif

    Material material;

    material.id = blockId;

    vec4 albedoTexture   = texelFetch(ATLAS_ALBEDO,   textureCoords, 0);
    vec4 normalTexture   = texelFetch(ATLAS_NORMAL,   textureCoords, 0);
    vec4 specularTexture = texelFetch(ATLAS_SPECULAR, textureCoords, 0);

    albedoTexture.rgb *= tint;

    material.opacity = albedoTexture.a;

    vec3 albedo = albedoTexture.rgb;

    #if TONEMAP == ACES && IS_RGB
        albedo = srgbToAP1Albedo(albedo);
    #else
        albedo = srgbToLinear(albedo);
    #endif

    #if WHITE_WORLD == 1
        albedo = vec3(1.0);
    #endif

    #if IS_SPECTRAL
        material.albedo = srgbToSpectrum(albedo, wavelength);
    #else
        material.albedo = albedo;
    #endif

    material.F0         = specularTexture.g;
	material.ao 		= 1.0;
	material.alpha      = vec2(maxEps(pow2(saturate(1.0 - specularTexture.r))));
	material.emission   = specularTexture.a * 255.0 < 254.5 ? specularTexture.a : 0.0;
	//material.subsurface = specularTexture.b * 255.0 < 65.0  ? 0.0 : specularTexture.b;
	//material.porosity   = specularTexture.b * 255.0 > 64.0  ? 0.0 : specularTexture.b;

    #if HARDCODED_EMISSIVES == 1
        if(isEmissive(blockId)) material.emission = 1.0;
    #endif

    material.normal          = hitNormal;
    material.tangentNormal   = hitNormal;
    material.geometricNormal = hitNormal;
    
    if(all(greaterThan(normalTexture, vec4(EPS)))) {
        material.normal.xy     = normalTexture.xy * 2.0 - 1.0;
	    material.normal.z      = sqrt(1.0 - saturate(dot(material.normal.xy, material.normal.xy)));
        material.tangentNormal = normalize(material.normal);
	    material.normal        = normalize(calculateTBN(hitNormal) * material.normal);

        #if MATERIAL_AO != 0
            material.ao = normalTexture.b;
        #endif
    }

    #if IS_SPECTRAL
        if(isGlass(blockId)) {
            material.N     = sellmeier(N_BK7HT, wavelength);
            material.F0    = iorToF0(material.N);
            material.alpha = vec2(1e-8);
            return material;
        }
    #endif

    if(material.F0.r * maxFloat8 > 229.5) {
        mat2x3 hcm = getHardcodedMetal(material);

        #if IS_RGB
            material.N = hcm[0], material.K = hcm[1];
        #else
            material.N = srgbToSpectrum(hcm[0], wavelength);
            material.K = srgbToSpectrum(hcm[1], wavelength);
        #endif
    } else {
        float ior = f0ToIOR(material.F0);

        #if IS_RGB
            material.N = vec3(ior);
            material.K = vec3(0.0);
        #else
            material.N = ior;
            material.K = 0.0;
        #endif
    }

    return material;
}

#if RENDER_MODE == 0
    Material getRasterMaterial(vec2 coords) {
        uvec4 dataTexture = texelFetch(RASTER_DATA_BUFFER, ivec2(coords * viewSize), 0);
        Material material;

        material.albedo = (uvec3(dataTexture.y) >> uvec3(0, 8, 16) & 255u) * rcpMaxFloat8;

        #if TONEMAP == ACES
            material.albedo = srgbToAP1Albedo(material.albedo);
        #else
            material.albedo = srgbToLinear(material.albedo);
        #endif

        #if WHITE_WORLD == 1
            material.albedo = vec3(1.0);
        #endif

        material.F0         = (dataTexture.x >> 16u & 255u) * rcpMaxFloat8;
        material.ao         = (dataTexture.x & 255u) * rcpMaxFloat8;
        material.alpha      = vec2((dataTexture.y >> 24u & 255u) * rcpMaxFloat8);
        material.emission   = (dataTexture.x >> 8u  & 255u) * rcpMaxFloat8;
        material.subsurface = (dataTexture.x >> 24u & 255u) * rcpMaxFloat8;

        #if MATERIAL_AO == 0
            material.ao = 1.0;
        #endif

        material.normal          = normalize(decodeUnitVector((uvec2(dataTexture.w) >> uvec2(0, 16) & 65535u) * rcpMaxFloat16));
        material.geometricNormal = normalize(decodeUnitVector((uvec2(dataTexture.z) >> uvec2(2, 17) & 32767u) * rcpMaxFloat15));

        material.parallaxSelfShadowing = float(dataTexture.z & 1u);

        //translucent = float(dataTexture.z >> 1u & 1u);

        if(material.F0.r * maxFloat8 > 229.5) {
            mat2x3 hcm = getHardcodedMetal(material);
            material.N = hcm[0], material.K = hcm[1];
        } else {
            material.N = vec3(f0ToIOR(material.F0));
            material.K = vec3(0.0);
        }

        return material;
    }
#endif
