/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

#if defined STAGE_VERTEX

	out vec2 textureCoords;

	void main() {
		textureCoords = gl_Vertex.xy;
		gl_Position   = vec4(gl_Vertex.xy * 2.0 - 1.0, 1.0, 1.0);
	}

#elif defined STAGE_FRAGMENT

	/* RENDERTARGETS: 0 */

	layout (location = 0) out vec3 lighting;

	in vec2 textureCoords;

	#include "/settings.glsl"
	#include "/include/common.glsl"

	#include "/include/atmospherics/constants.glsl"

	#include "/include/voxels/voxelization.glsl"

	#include "/include/materials/material.glsl"

	#include "/include/post/exposure.glsl"

	uniform sampler2D colortex2;

	#if RENDER_MODE == 1
		#if TEMPORAL_ACCUMULATION == 1
			#include "/include/atmospherics/celestial_paths.glsl"

		#endif
	#endif

	#if RENDER_MODE == 0
		#include "/include/atmospherics/illuminance_ssbo.glsl"

		#if defined WORLD_OVERWORLD
			#include "/include/fragment/belmu/fresnel.glsl"
			#include "/include/fragment/hammon.glsl"

			#include "/include/atmospherics/celestial_bodies.glsl"

			#include "/include/fragment/shadowmap.glsl"
			#include "/include/fragment/belmu/brdf.glsl"
			#include "/include/fragment/belmu/cook_torrance.glsl"
			#include "/include/fragment/belmu/subsurface.glsl"

			vec3 sampleAtmosphere(vec3 viewPosition, vec3 sceneDirection) {
				vec4 clouds = vec4(0.0, 0.0, 0.0, 1.0);
				#if CLOUDS_LAYER0_ENABLED == 1 || CLOUDS_LAYER1_ENABLED == 1
					vec3 cloudsBuffer = texture(CLOUDS_BUFFER, textureCoords).rgb;

					clouds.rgb = cloudsBuffer.r * directIlluminance + cloudsBuffer.g * skyIlluminance;
					clouds.a   = cloudsBuffer.b;
				#endif

				vec3 atmosphere = texture(ATMOSPHERE_BUFFER, projectSphere(sceneDirection)).rgb;
			
				atmosphere += clamp16(physicalSun(sceneDirection, sunRadiance));
				atmosphere += physicalMoon(sceneDirection, sunIrradiance);

				atmosphere += computeStarfield(viewPosition, sceneDirection);

				return atmosphere * clouds.a + clouds.rgb;
			}
		#endif
	#endif

	#include "/include/voxels/raytracer.glsl"

	#include "/include/fragment/cache/irradiance_cache.glsl"

	void main() {
		/*
			vec3 screenPosition2 = vec3(textureCoords, texture(depthtex0, textureCoords).r);
			vec3 sceneDirection2 = normalize(mat3(gbufferModelViewInverse) * screenToView(screenPosition2, false));

			VoxelIntersection hitCache = raytraceVoxel2(cameraPosition, sceneDirection2, 256, true, true);
			lighting = readCurrentCacheEntry(ivec3((hitCache.position - hitCache.normal * 1e-4)), getFaceIndex(hitCache.normal));
			return;
			*/

		#if DEBUG_ALBEDO == 1 || DEBUG_NORMALS == 1 || DEBUG_HIT_POSITION == 1
			lighting = texture(IRRADIANCE_BUFFER, textureCoords).rgb;
			return;
		#endif

		#if RENDER_MODE == 1 || ATROUS_FILTER == 0
			vec3 radiance = texture(IRRADIANCE_BUFFER, textureCoords).rgb;
		#else
			vec3 radiance = texture(LIGHTING_BUFFER, textureCoords).rgb;
		#endif

		float exposure = computeExposure(texelFetch(HISTORY_BUFFER, ivec2(0), 0).a);

		#if RENDER_MODE == 0
			vec3 direct = vec3(0.0);

			#if defined WORLD_OVERWORLD
				float depth = texture(depthtex0, textureCoords).r;

				vec3 screenPosition = vec3(textureCoords, depth);
				vec3 viewPosition   = screenToView(screenPosition, false);

				if(depth == 1.0) {
					vec3 sceneDirection = normalize(viewToScene(viewPosition));

					lighting  = sampleAtmosphere(viewPosition, sceneDirection);

					#if TAA == 1
						lighting *= exposure;
						lighting  = reinhard(lighting);
					#endif
					return;
				}
			#endif

			Material material = getRasterMaterial(textureCoords);

			bool isMetal = material.F0.r * maxFloat8 > 229.5;

			#if defined WORLD_OVERWORLD
				vec3 scenePosition = viewToScene(viewPosition);

				vec3 shadowRayDirection = generateConeVector(shadowLightVector, rand2F(), shadowLightAngularRadius);
                
                vec3 diffuse  = isMetal ? vec3(0.0) : evaluateMicrosurfaceOpaqueDiffuse(material, shadowRayDirection);
				vec3 specular = computeSpecular(material, -normalize(scenePosition), shadowRayDirection);

                float subsurfaceDepth = 0.0;
                vec3 shadowmap = calculateShadowMapping(scenePosition, material.geometricNormal, subsurfaceDepth);

				vec3 subsurface = isMetal ? vec3(0.0) : subsurfaceScatteringApprox(material, -normalize(viewPosition), shadowVec, subsurfaceDepth);

				direct  = diffuse + specular;
				direct *= shadowmap * material.parallaxSelfShadowing;
				direct += material.albedo * subsurface;
				direct *= directIlluminance;
			#endif

			vec3 ambient = isMetal ? vec3(0.0) : vec3(0.2);

			float emissiveness = material.emission * EMISSIVE_INTENSITY * EMISSIVE_INTENSITY_MULTIPLIER;

			lighting = direct + material.albedo * (radiance + emissiveness + ambient);
			//lighting = radiance;

			#if TAA == 1
				lighting *= exposure;
				lighting  = reinhard(lighting);
    		#endif
		#else
			#if TEMPORAL_ACCUMULATION == 1 && DEBUG_ALBEDO == 0 && DEBUG_NORMALS == 0 && DEBUG_HIT_POSITION == 0
				if(hideGUI == 0) {
					vec3 screenPosition = vec3(textureCoords, texture(depthtex0, textureCoords).r);
					vec3 sceneDirection = normalize(mat3(gbufferModelViewInverse) * screenToView(screenPosition, false));

					VoxelIntersection hitAlbedo = raytraceVoxel(cameraPosition, sceneDirection, MAX_RAYTRACE_STEPS, true, true);

					if(hitAlbedo.intersect) {
						vec3 albedo = texelFetch(ATLAS_ALBEDO, hitAlbedo.textureCoords, 0).rgb * unpackTint(hitAlbedo.voxel.packedData);

						#if TONEMAP == ACES
							albedo = srgbToAP1Albedo(albedo);
						#else
							albedo = srgbToLinear(albedo);
						#endif

						lighting = albedo / exposure;
					}

					vec3 sunPosition  = calculateSunPosition (LATITUDE, LONGITUDE);
					vec3 moonPosition = calculateMoonPosition(LATITUDE, LONGITUDE);

					vec3 sunDirection  = normalize(sphericalToCartesian(sunPosition ));
					vec3 moonDirection = normalize(sphericalToCartesian(moonPosition));

					lighting += (dot(sceneDirection, sunDirection ) < cos(sunAngularRadius ) ? vec3(0.0) : vec3(0.87, 0.63, 0.08)) / exposure;
					lighting += (dot(sceneDirection, moonDirection) < cos(moonAngularRadius) ? vec3(0.0) : vec3(1.0             )) / exposure;
					return;
				}
			#endif

			lighting = radiance;
		#endif
	}
	
#endif
