/***********************************************/
/*           Copyright (C) 2024 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

#include "/settings.glsl"

#if defined STAGE_VERTEX

	out vec2 textureCoords;

	void main() {
		textureCoords = gl_Vertex.xy;
		gl_Position   = vec4(gl_Vertex.xy * 2.0 - 1.0, 1.0, 1.0);
	}

#elif defined STAGE_FRAGMENT

	#if !(RENDER_MODE == 0 && RESTIR_GI == 1 && DEBUG_ALBEDO == 0 && DEBUG_NORMALS == 0 && DEBUG_HIT_POSITION == 0)
		/* RENDERTARGETS: 1 */

		layout (location = 0) out vec3 radianceOut;
	#endif

	in vec2 textureCoords;

	#include "/include/common.glsl"

	#include "/include/voxels/voxelization.glsl"
	
	#include "/include/materials/material.glsl"

	#include "/include/voxels/raytracer.glsl"

	#if DEBUG_ALBEDO == 0 && DEBUG_NORMALS == 0 && DEBUG_HIT_POSITION == 0
		#include "/include/fragment/belmu/fresnel.glsl"

		#if RENDER_MODE == 0
			#include "/include/fragment/hammon.glsl"
		#endif

		#include "/include/atmospherics/constants.glsl"
		#include "/include/atmospherics/celestial_bodies.glsl"
	
		#if RENDER_MODE == 0
			#include "/include/atmospherics/illuminance_ssbo.glsl"

			#include "/include/fragment/shadowmap.glsl"

			#include "/include/fragment/belmu/brdf.glsl"

			#include "/include/fragment/cache/irradiance_cache.glsl"

			#include "/include/fragment/belmu/pathtracer.glsl"

			#if RESTIR_GI == 1
				#include "/include/fragment/restir/common.glsl"
				#include "/include/fragment/restir/temporal_reuse.glsl"
			#endif
		#else
			#include "/include/utility/complex.glsl"

			#include "/include/fragment/render/fresnel.glsl"

			//#include "/include/camera/lens/thick_lens.glsl"

			#include "/include/atmospherics/spectral_atmosphere.glsl"
			#include "/include/atmospherics/celestial_paths.glsl"

			#include "/include/fragment/render/pathtracer.glsl"
		#endif
	#endif

	void main() {
		vec3 screenPosition = vec3(textureCoords, texture(depthtex0, textureCoords).r);
		vec3 viewPosition   = screenToView(screenPosition, false);
    	vec3 sceneDirection = normalize(mat3(gbufferModelViewInverse) * viewPosition);

		#if DEBUG_ALBEDO == 1 || DEBUG_NORMALS == 1 || DEBUG_HIT_POSITION == 1
		
		    /* Debug */
			VoxelIntersection hit = raytraceVoxel(cameraPosition, sceneDirection, MAX_RAYTRACE_STEPS, true, true);

			if(hit.intersect) {
				#if DEBUG_ALBEDO == 1
					radianceOut = getVoxelMaterial(hit.voxel.packedData, hit.textureCoords, hit.position, hit.normal, 0.0).albedo;

				#elif DEBUG_NORMALS == 1
					radianceOut = getVoxelMaterial(hit.voxel.packedData, hit.textureCoords, hit.position, hit.normal, 0.0).normal * 0.5 + 0.5;

				#elif DEBUG_HIT_POSITION == 1
					radianceOut = (hit.position - ivec3(floor(hit.position)) - vec3(0.5)) * 0.5 + 0.5;
				#endif
			}

		#else

			#if RENDER_MODE == 0

				float depth = screenPosition.z;
				if(depth == 1.0) return;

				vec3 worldPosition = viewToWorld(viewPosition);

				vec3 radianceDiffuse = vec3(0.0), radianceSpecular = vec3(0.0);

				vec3 samplePositionDiffuse = vec3(0.0), samplePositionSpecular = vec3(0.0);
				vec3 sampleNormalDiffuse   = vec3(0.0), sampleNormalSpecular   = vec3(0.0);

				pathtraceBelmuBRDFDiffuse (worldPosition, sceneDirection, directIlluminance, radianceDiffuse , samplePositionDiffuse , sampleNormalDiffuse );
				pathtraceBelmuBRDFSpecular(worldPosition, sceneDirection, directIlluminance, radianceSpecular, samplePositionSpecular, sampleNormalSpecular);

				#if RESTIR_GI == 0

					radianceOut = radianceDiffuse + radianceSpecular;

				#else

					Material material = getRasterMaterial(textureCoords);

					writeVisiblePositionCurrent(worldPosition, ivec2(gl_FragCoord.xy));

					Reservoir reservoirDiffuse  = Reservoir(radianceDiffuse , samplePositionDiffuse , sampleNormalDiffuse , 0.0, 1);
					Reservoir reservoirSpecular = Reservoir(radianceSpecular, samplePositionSpecular, sampleNormalSpecular, 0.0, 1);

					float diffuseTarget  = computeIntegrand(radianceDiffuse , samplePositionDiffuse , material, worldPosition, material.normal);
					float specularTarget = computeIntegrand(radianceSpecular, samplePositionSpecular, material, worldPosition, material.normal);

					float diffusePDF  = computeDiffusePDF (samplePositionDiffuse , material, worldPosition, material.normal);
					float specularPDF = computeSpecularPDF(samplePositionSpecular, material, worldPosition, material.normal);

					reservoirDiffuse.weight  = diffusePDF  <= 0.0 ? 0.0 : diffuseTarget  / diffusePDF;
					reservoirSpecular.weight = specularPDF <= 0.0 ? 0.0 : specularTarget / specularPDF;
					
					vec3 prevPosition = screenPosition - getVelocity(screenPosition);
					ReSTIR_Temporal_Reuse(reservoirDiffuse, prevPosition, material, worldPosition, material.normal, depth);

					Reservoir reservoir = pickReservoirMIS(reservoirDiffuse, reservoirSpecular, material, worldPosition, material.normal);

					writeSpatialReservoir(reservoir, ivec2(gl_FragCoord.xy));

				#endif

			#else

				#if TEMPORAL_ACCUMULATION == 1
					if(hideGUI == 0)
						return;
				#endif

				bool canRender = true;
				#if CHECKERBOARD_RENDERING == 1
					ivec2 coords    = ivec2(gl_FragCoord.xy / int(viewResolution.x * 0.1));
						canRender = (coords.x + coords.y & 1) == (frameCounter & 1);
				#endif

				/* CIE Importance Sampling */
				vec3 history   = vec3(0.0);
				const int size = 4;

				for(int x = -size; x <= size; x++) {
					for(int y = -size; y <= size; y++) {
						history += texture(IRRADIANCE_BUFFER, textureCoords + vec2(x, y) * texelSize).rgb;
					}
				}
				vec3 cmfWeights = mix(saturate(history / dot(vec3(1.0), history)), vec3(1.0 / 3.0), history == vec3(0.0) ? 1.0 : 0.2);

				for(int pathCount = 0; pathCount < MAX_PT_SAMPLES; pathCount++) {
					float wavelength = binarySearchCIE(0.0, 1.0, randF(), cmfWeights) * 441.0 + 390.0;

					airIOR = airDispersion(wavelength, 15.0, 101325.0, 0.0, 400.0);

					vec3 sensorPosition = cameraPosition;

					float throughput = 1.0;

					//bool rejected = !traceLensSystem(textureCoords, wavelength, throughput, sensorPosition, sceneDirection);

					if (canRender) {
						radianceOut += evaluateHeitzBSDF(throughput, sensorPosition, sceneDirection, wavelength, cmfWeights) / MAX_PT_SAMPLES;
					}
				}
				
			#endif
		#endif
	}
	
#endif
