/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

#include "/settings.glsl"

#if MANUAL_CAMERA == 0 && EXPOSURE == 2

    #include "/include/uniforms.glsl"
    #include "/include/utility/math.glsl"
    #include "/include/utility/color.glsl"

    layout (r32ui) uniform uimage1D histogram;

    #if defined HISTOGRAM_BUILD_PASS

        layout (local_size_x = 16, local_size_y = 8) in;

        shared uint histogramLocal[HISTOGRAM_BINS];

        int getBinFromLuminance(float luminance) {
            return clamp(int(round((log(luminance) * rcpLuminanceRange - (minLuminance * rcpLuminanceRange)) * HISTOGRAM_BINS)), 0, HISTOGRAM_BINS - 1);
        }

        void main() {
            histogramLocal[gl_LocalInvocationIndex] = 0u;

            memoryBarrierShared();
            barrier();

            if (all(lessThan(gl_GlobalInvocationID.xy, textureSize(BLOOM_BUFFER, 0)))) {
                float luminance = luminance(texelFetch(BLOOM_BUFFER, ivec2(gl_GlobalInvocationID.xy * 0.5), 0).rgb);

                if (!isnan(luminance) && !isinf(luminance) && luminance > 0.0) {
                    atomicAdd(histogramLocal[getBinFromLuminance(luminance)], 1u);
                }
            }

            memoryBarrierShared();
            barrier();

            imageAtomicAdd(histogram, int(gl_LocalInvocationIndex), histogramLocal[gl_LocalInvocationIndex]);
        }

    #elif defined HISTOGRAM_MEDIAN_PASS

        layout (local_size_x = 1) in;

        const ivec3 workGroups = ivec3(1, 1, 1);

        uniform usampler1D histogramSampler;

        float getLuminanceFromBin(int bin) {
            return exp((bin * rcp(HISTOGRAM_BINS)) * luminanceRange + minLuminance);
        }

        int getClosestBinToMedian() {
            uint totalDensity = 0;
            for(int bin = 0; bin < HISTOGRAM_BINS; bin++) {
                totalDensity += texelFetch(histogramSampler, bin, 0).r;
            }

            uint median = totalDensity / 2;
            uint cumulativeDensity = 0;

            for(int bin = 0; bin < HISTOGRAM_BINS; bin++) {
                cumulativeDensity += texelFetch(histogramSampler, bin, 0).r;

                if(cumulativeDensity >= median) return bin;
            }

            return HISTOGRAM_BINS - 1;
        }

        void main() {
            int   medianBin        = getClosestBinToMedian();
            float averageLuminance = getLuminanceFromBin(medianBin);

            imageStore(histogram, 0, uvec4(floatBitsToUint(averageLuminance), 0, 0, 0));
        }

    #endif

#else

    layout (local_size_x = 1) in;

    void main() {
        return;
    }

#endif
