/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

#include "/settings.glsl"

#if RENDER_MODE == 1 || CLOUDS_LAYER0_ENABLED == 0 && CLOUDS_LAYER1_ENABLED == 0 || DEBUG_ALBEDO == 1 || DEBUG_NORMALS == 1 || DEBUG_HIT_POSITION == 1 || !defined WORLD_OVERWORLD
    #include "/programs/discard.glsl"
#else
    #if defined STAGE_VERTEX

        out vec2 textureCoords;

        void main() {
            textureCoords = gl_Vertex.xy;
            gl_Position   = vec4(gl_Vertex.xy * 2.0 - 1.0, 1.0, 1.0);
        }

    #elif defined STAGE_FRAGMENT

        /* RENDERTARGETS: 5 */

        layout (location = 0) out vec3 clouds;

        in vec2 textureCoords;

        #include "/include/common.glsl"

        #include "/include/utility/sampling.glsl"
        
        #include "/include/atmospherics/clouds.glsl"

        float find4x4MaximumDepth(vec2 coords) {
            coords *= viewSize;

            return maxOf(vec4(
                texelFetch(depthtex0, ivec2(coords) + ivec2( 2,  2), 0).r,
                texelFetch(depthtex0, ivec2(coords) + ivec2(-2,  2), 0).r,
                texelFetch(depthtex0, ivec2(coords) + ivec2(-2, -2), 0).r,
                texelFetch(depthtex0, ivec2(coords) + ivec2( 2, -2), 0).r
            ));
        }

        void main() {
            clouds = vec3(0.0, 0.0, 1.0);

            if(find4x4MaximumDepth(textureCoords) < 1.0) return;

            vec3 viewPosition       = screenToView(vec3(textureCoords, 1.0), false);
            vec3 cloudsRayDirection = mat3(gbufferModelViewInverse) * normalize(viewPosition);

            vec4 layer0 = vec4(0.0, 0.0, 1.0, 1e9);
            vec4 layer1 = vec4(0.0, 0.0, 1.0, 1e9);

            #if CLOUDS_LAYER0_ENABLED == 1
                layer0 = estimateCloudsScattering(cloudLayer0, cloudsRayDirection);
            #endif

            #if CLOUDS_LAYER1_ENABLED == 1
                layer1 = estimateCloudsScattering(cloudLayer1, cloudsRayDirection);
            #endif

            float distanceToClouds = min(layer0.a, layer1.a);

            clouds.rg = layer0.rg + layer1.rg * layer0.b;
            clouds.b  = layer0.b  * layer1.b;

            /* Aerial Perspective */
            clouds.rgb = mix(vec3(0.0, 0.0, 1.0), clouds.rgb, quinticStep(0.0, 1.0, sqrt(max0(exp(-5e-5 * distanceToClouds)))));

            /* Reprojection */
            vec2  prevPosition = reproject(viewPosition, distanceToClouds, CLOUDS_WIND_SPEED * frameTime * windDir).xy;
            float prevDepth    = texture(depthtex0, prevPosition.xy).r;

            if(saturate(prevPosition.xy) == prevPosition.xy && prevDepth >= handDepth) {
                vec3 history = max0(textureCatmullRom(CLOUDS_BUFFER, prevPosition.xy).rgb);

                const float centerWeightStrength = CLOUDS_SCALE == 100 ? 0.6 : 0.1;

                vec2 pixelCenterDist = 1.0 - abs(2.0 * fract(prevPosition.xy * viewSize) - 1.0);
                float centerWeight   = sqrt(pixelCenterDist.x * pixelCenterDist.y) * centerWeightStrength + (1.0 - centerWeightStrength);
                
                float velocityWeight = saturate(exp(-1.0 * length(cameraPosition - previousCameraPosition)));

                float weight = clamp(centerWeight * velocityWeight, 0.0, 0.97);

                clouds.rgb = max0(mix(clouds.rgb, history, weight));
            }
        }
        
    #endif
#endif
