/***********************************************/
/*           Copyright (c) 2025 Belmu          */
/*             All Rights Reserved             */
/***********************************************/

/*
    [Credits]:
        Xirreal - providing base code for voxelization, thanks to them!
*/

#include "/settings.glsl"
#include "/include/common.glsl"

#if defined STAGE_VERTEX

    #define attribute in
    attribute vec2 mc_midTexCoord;
    attribute vec3 at_midBlock;
    attribute vec3 mc_Entity;

    out VertexData {
        int  blockId;
        vec2 textureCoords;
        vec2 midCoords;
        vec3 midBlock;
        vec3 normal;
        vec3 scenePosition;
        vec4 tint;
    } vs;

    void main() {
        vs.blockId = int(mc_Entity.x);

        vs.midCoords     = mc_midTexCoord;
        vs.textureCoords = gl_MultiTexCoord0.xy;

        vs.midBlock      = at_midBlock / 64.0;
        vs.normal        = gl_Normal;
        vs.scenePosition = gl_Vertex.xyz;

        vs.tint = gl_Color;

        #if RENDER_MODE == 0 && DEBUG_ALBEDO == 0 && DEBUG_NORMALS == 0 && DEBUG_HIT_POSITION == 0
            gl_Position     = ftransform();
            gl_Position.xyz = distortShadowSpace(gl_Position.xyz);
        #endif
    }

#elif defined STAGE_GEOMETRY

    layout (triangles) in;

    in VertexData {
        int  blockId;
        vec2 textureCoords;
        vec2 midCoords;
        vec3 midBlock;
        vec3 normal;
        vec3 scenePosition;
        vec4 tint;
    } vs[3];

    #if RENDER_MODE == 0 && DEBUG_ALBEDO == 0 && DEBUG_NORMALS == 0 && DEBUG_HIT_POSITION == 0
        layout (triangle_strip, max_vertices = 4) out;

        out vec2 textureCoords;
        out vec4 tint;
    #else
        layout (points, max_vertices = 1) out;
    #endif

    uniform sampler2D gtexture;

    uniform int renderStage;

    #if SPHERES_MODE == 1 && RENDER_MODE == 1
        const vec3 glassColors[] = vec3[](
            vec3(0.090, 0.541, 0.910),
            vec3(0.090, 0.949, 0.027),
            vec3(0.949, 0.118, 0.075),
            vec3(0.949, 0.600, 0.075),
            vec3(0.929, 0.047, 0.886),
            vec3(0.969, 0.875, 0.027)
        );
    #endif

    #include "/include/voxels/voxelization.glsl"

    void main() {
        #if RENDER_MODE == 0 && DEBUG_ALBEDO == 0 && DEBUG_NORMALS == 0 && DEBUG_HIT_POSITION == 0

            for(int vertex = 0; vertex < 3; vertex++) {
                gl_Position   = gl_in[vertex].gl_Position;
                textureCoords = vs[vertex].textureCoords;
                tint          = vs[vertex].tint;
                EmitVertex();
            }
            EndPrimitive();

        #endif

        if(vs[0].blockId == 0 || renderStage == MC_RENDER_STAGE_ENTITIES) return;

        vec3  voxelCenter   = (vs[0].midBlock + vs[1].midBlock + vs[2].midBlock + vs[0].scenePosition + vs[1].scenePosition + vs[2].scenePosition) / 3.0;
        ivec3 voxelPosition = ivec3(floor(voxelCenter + cameraPosition));

        if(!isInVoxelBounds(voxelPosition)) return;

        ivec2 atlasResolution    = textureSize(gtexture, 0);
        vec2  atlasAspectCorrect = vec2(1.0, atlasResolution.x / atlasResolution.y);

        // Thanks Zombye for the help with fixing maximum edge calculation for versions above 1.16
        float maxEdge        = max(maxOf(abs(vs[0].scenePosition - vs[1].scenePosition)), maxOf(abs(vs[0].scenePosition - vs[2].scenePosition)));
        float tileSize       = max(maxOf(abs(vs[0].textureCoords - vs[0].midCoords) / atlasAspectCorrect), maxOf(abs(vs[1].textureCoords - vs[1].midCoords) / atlasAspectCorrect)) / maxEdge;
        float tileResolution = 2.0 * tileSize * maxOf(atlasResolution);
        ivec2 tileIndex      = ivec2(round((vs[0].midCoords - tileSize * atlasAspectCorrect) * atlasResolution / round(tileResolution)));

        uint facePriority = uint(dot(max0(vs[0].normal), vec3(4, 5, 3)) + dot(max0(-vs[0].normal), vec3(2, 0, 1)));

        vec3 tint = vs[0].tint.rgb;
        #if SPHERES_MODE == 1 && RENDER_MODE == 1
            if(vs[0].blockId == SPHERE_1) {
                tint = glassColors[int(rand(abs((voxelCenter + cameraPosition).xz * voxelCenter.y)) * 5.0)];
            }
        #endif

        writeVoxel(voxelPosition, tint, vs[0].blockId, facePriority, int(ceil(tileResolution)), tileIndex);

        EmitVertex();
        EndPrimitive();
    }

#elif defined STAGE_FRAGMENT

    #if RENDER_MODE == 0 && DEBUG_ALBEDO == 0 && DEBUG_NORMALS == 0 && DEBUG_HIT_POSITION == 0

        /* RENDERTARGETS: 0 */

        layout(location = 0) out vec4 shadowmap;

        in vec2 textureCoords;
        in vec4 tint;

        uniform sampler2D gtexture;

        void main() {
            vec4 albedoTex = texture(gtexture, textureCoords) * tint;
            if(albedoTex.a < 0.102) discard;

            #if WHITE_WORLD == 1
                albedoTex.rgb = vec3(1.0);
            #endif

            shadowmap = albedoTex;
        }

    #else

        void main() {
            discard; return;
        }

    #endif
    
#endif
